unit TurboSprite;

interface
                     
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, ExtCtrls, Forms, Dialogs,
  ColorPalette, Grafix, ColorIndexSelector, DsgnIntf, DIBFileLoader, DIBCanvas;

type
  TSpinType = ( spinNone, spinClockWise, spinCounterClockWise );

  TSprite = class;
  TDIB = class;

  ETurboSprite = class( Exception );

  TCollisionEvent = procedure( Sender: TObject; Sprite, TargetSprite: TSprite ) of object;
  TSpriteClickEvent = procedure( Sender: TObject; Button: TMouseButton; Shift: TShiftState; Sprite: TSprite; X, Y: integer ) of object;
  TMoveProc = procedure( sprite: TSprite );

{ TRectHolder }
  TRectHolder = class( TObject )
  private
  protected
  public
    rect: TRect;
    constructor Create( rectangle: TRect );
  end;

{ TBM - My TBitmapInfo }
  TBM = record
     bmheader: TBitmapInfoHeader;
     bmiColors: array[0..255] of TRGBQuad;
  end;

{ TSpriteSurface }
  TSpriteSurface = class( TGraphicControl )
  private
    procedure CreateDIB;
    procedure DestroyDIB;
  protected
    FColorPalette: TColorPalette;
    FStretch: boolean;
    FDIBCanvas: TDIBCanvas;
    FDIBCreated: boolean;
    FDIBDC: hDC;
    FPaletteOld: HPALETTE;
    FBitmapInfo: TBM;
    FDIBHandle: HBITMAP;
    FDIBBits: pointer;
    FOldBitmap: HBITMAP;
    FAutoBlank: boolean;
    FAutoColor: byte;
    FBeforeSpriteRender: TNotifyEvent;
    FAfterSpriteRender: TNotifyEvent;
    FResized: boolean;
    FGDICanvas: TCanvas;
    FForce: boolean;
    FDirty: boolean;
    FOffsetX, FOffsetY: integer;
    FLogicalWidth, FLogicalHeight: integer;
    FWrapHorz, FWrapVert: boolean;
    lstEngines: TList;
    FLockLogical: boolean;
    FBackground: TDIB;
    FMouseDown: TMouseEvent;
    FMouseMove: TMouseMoveEvent;
    FAbsMouseMove: TMouseMoveEvent;
    FMouseUp: TMouseEvent;
    FSpriteClicked: TSpriteClickEvent;
    SortEngines: boolean;
    FOnResize: TNotifyEvent;
    procedure Loaded; override;
    procedure Notification( AComponent: TComponent; Operation: TOperation ); override;
    procedure Paint; override;
    procedure SetBounds( ALeft, ATop, AWidth, AHeight: Integer ); override;
    procedure setColorPalette( pal: TColorPalette );
    procedure setAutoBlank( b: boolean );
    procedure setAutoBlankColor( n: byte );
    procedure MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;
    procedure MouseMove( Shift: TShiftState; X, Y: Integer ); override;
    procedure MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: Integer ); override;
    function getMouseX( X: integer ): integer;
    function getMouseY( Y: integer ): integer;
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
    procedure synchColorEntries( StartIndex, NbrOfEntries: integer );
    function randomPoint: TPoint;
    procedure DIBRefresh;
    property GDICanvas: TCanvas read FGDICanvas;
  published
    property Align;
    property AutoBlank: boolean read FAutoBlank write setAutoBlank;
    property AutoBlankColorIndex: byte read FAutoColor write setAutoBlankColor;
    property BackgroundDIB: TDIB read FBackground write FBackground;
    property ColorPalette: TColorPalette read FColorPalette write setColorPalette;
    property DIBCanvas: TDIBCanvas read FDIBCanvas;
    property DirtyRectangles: boolean read FDirty write FDirty;
    property LockLogicalSize: boolean read FLockLogical write FLockLogical;
    property LogicalWidth: integer read FLogicalWidth write FLogicalWidth;
    property LogicalHeight: integer read FLogicalHeight write FLogicalHeight;
    property OffsetX: integer read FOffsetX write FOffsetX;
    property OffsetY: integer read FOffsetY write FOffsetY;
    property Stretch: boolean read FStretch write FStretch default false;
    property WrapHorizontal: boolean read FWrapHorz write FWrapHorz;
    property WrapVertical: boolean read FWrapVert write FWrapVert;
    property BeforeSpriteRender: TNotifyEvent read FBeforeSpriteRender write FBeforeSpriteRender;
    property AfterSpriteRender: TNotifyEvent read FAfterSpriteRender write FAfterSpriteRender;
    property OnMouseDown: TMouseEvent read FMouseDown write FMouseDown;
    property OnMouseMove: TMouseMoveEvent read FMouseMove write FMouseMove;
    property OnMouseMoveAbs: TMouseMoveEvent read FAbsMouseMove write FAbsMouseMove;
    property OnMouseUp: TMouseEvent read FMouseUp write FMouseUp;
    property OnClick: TSpriteClickEvent read FSpriteClicked write FSpriteClicked;
    property OnResize: TNotifyEvent read FOnResize write FOnResize;
  end;

{ TSpriteEngine }
  TSpriteEngine = class( TComponent )
  private
    lstSprites: TList;
    lstNewSprites: TList;
    lstDeadSprites: TList;
    lstDirty: TList;
    FSpriteSurface: TSpriteSurface;
    FExpectedSprites: integer;
    FCollision: TCollisionEvent;
    bDrawAll: boolean;
    lstEngines: TList;
    FPri: integer;
    FClick: boolean;
  protected
    function GetSprite( n: integer ): TSprite;
    function GetSpriteCount: integer;
    procedure SetExpectedSprites( n: integer );
    procedure Notification( AComponent: TComponent; Operation: TOperation ); override;
    procedure ProcessSprites;
    procedure RenderSprites;
    procedure DetectCollisions;
    procedure CollisionDetection( se: TSpriteEngine );
    procedure setSpriteSurface( ss: TSpriteSurface );
    procedure setPri( n: integer );
  public
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
    procedure AddCollisionEngine( engine: TSpriteEngine );
    procedure AddSprite( spr: TSprite );
    procedure ChangeSpritePriority( spr: TSprite; n: integer );
    procedure RemoveCollisionEngine( engine: TSpriteEngine );
    procedure RemoveSprite( spr: TSprite );
    procedure removeAllSprites;
    property Sprite[n: integer]: TSprite read GetSprite;
    property SpriteCount: integer read GetSpriteCount;
  published
    property ClickSprites: boolean read FClick write FClick default true;
    property SpriteSurface: TSpriteSurface read FSpriteSurface write setSpriteSurface;
    property ExpectedSprites: integer read FExpectedSprites write SetExpectedSprites;
    property Priority: integer read FPri write setPri;
    property OnCollision: TCollisionEvent read FCollision write FCollision;
  end;

{ TSprite }
  TSprite = class( TObject )
  private
    bDummy: boolean;
    function getCorrectedAngle: integer;
  protected
    FMoved: boolean;
    FPri: integer;
    FVisible: boolean;
    FDirty: boolean;
    FWidth: integer;
    FHeight: integer;
    FWidth2: integer;
    FHeight2: integer;
    bKill: boolean;
    FMoveProc: TMoveProc;
    FMarginLeft, FMarginRight, FMarginTop, FMarginBottom: integer;
    FSpinType: TSpinType;
    FAngle: integer;
    OldAngle: integer;
    FSpinSpeed: integer;
    ptDestination: TPoint;
    ptLastDrawn: TPoint;       { The last position the sprite was drawn }
    FSpriteSurface: TSpriteSurface;
    FSpriteEngine: TSpriteEngine;
    FStopAtDest: boolean;
    FLastSpeed: single;
    FLastDest: TPoint;
    ptPhysical: TPoint;
    X, Y: single;
    FSpeed: single;
    FSpeedX, FSpeedY: single;
    FMaxSpeed: single;
    FFriction: single;
    FThrust: single;
    FMoveX, FMoveY: integer;
    FOrigin: TPoint;
    function GetBoundingRect: TRect; virtual;
    function GetCollisionRect: TRect; virtual;
    function GetDirtyRect: TRect; virtual;
    function getPosition: TPoint;
    procedure Move; virtual;
    procedure SetDestination( const pt: TPoint ); virtual;
    procedure setSpeed( const x: single ); virtual;
    procedure SetWidth( n: integer );
    procedure SetHeight( n: integer );
    procedure SetDead( b: boolean ); virtual;
    procedure RefreshBackground; virtual;
    procedure Render; virtual;
    procedure setAngle( n: integer ); virtual;
    procedure setPosition( pt: TPoint );
    property correctedAngle: integer read getCorrectedAngle;
  public
    constructor Create( MoveProc: TMoveProc );
    function destinationChanged: boolean;
    function speedChanged: boolean;
    function FudgedDistance( s: TSprite ): single;
    procedure Stop; virtual;
    property Angle: integer read FAngle write setAngle;
    property BoundingRect: TRect read GetBoundingRect;
    property CollisionRect: TRect read getCollisionRect;
    property Destination: TPoint read ptDestination write SetDestination;
    property Dead: boolean read bKill write SetDead;
    property Friction: single read FFriction write FFriction;
    property Height: integer read FHeight write SetHeight;
    property MarginLeft: integer read FMarginLeft write FMarginLeft;
    property MarginRight: integer read FMarginRight write FMarginRight;
    property MarginTop: integer read FMarginTop write FMarginTop;
    property MarginBottom: integer read FMarginBottom write FMarginBottom;
    property MaximumSpeed: single read FMaxSpeed write FMaxSpeed;
    property Moved: boolean read FMoved write bDummy;
    property MoveDirectionX: integer read FMoveX write FMoveX;
    property MoveDirectionY: integer read FMoveY write FMoveY;
    property Origin: TPoint read FOrigin write FOrigin;
    property PhysicalPosition: TPoint read ptPhysical;
    property Position: TPoint read getPosition write setPosition;
    property PositionX: single read X write X;
    property PositionY: single read Y write Y;
    property Priority: integer read FPri write FPri;
    property Speed: single read FSpeed write setSpeed;
    property SpeedX: single read FSpeedX write FSpeedX;
    property SpeedY: single read FSpeedY write FSpeedY;
    property SpinSpeed: integer read FSpinSpeed write FSpinSpeed;
    property SpinType: TSpinType read FSpinType write FSpinType;
    property SpriteSurface: TSpriteSurface read FSpriteSurface;
    property StopAtDestination: boolean read FStopAtDest write FStopAtDest;
    property Thrust: single read FThrust write FThrust;
    property Visible: boolean read FVisible write FVisible default TRUE;
    property Width: integer read FWidth write SetWidth;
  end;

{ TDIB }
  TDIB = class( TComponent )
  private
    nDummy: word;
    niDummy: integer;
    nDummyLong: longint;
    FDIBCanvas: TDIBCanvas;
    FFileName: TFileName;
    FFramesX, FFramesY: integer;
    FNumColors: word;
    FImage: TImage;
    FColorPalette: TColorPalette;
    FTrans: boolean;
    procedure setColorPalette( pal: TColorPalette );
  protected
    pDIBBits: pointer;
    procedure CleanUp; virtual;
    procedure ReadImage( Stream: TStream ); virtual;
    procedure WriteImage( Stream: TStream ); virtual;
    procedure DefineProperties( Filer: TFiler ); override;
    procedure SetFileName( const sName: TFileName ); virtual;
    procedure SetImage( im: TImage ); virtual;
    procedure LoadFromBitmap( bm: Graphics.TBitmap ); virtual;
    function GetDataLoaded: boolean;
    procedure SetDataLoaded( b: boolean );
    procedure Notification( AComponent: TComponent; Operation: TOperation ); override;
    procedure Loaded; override;
  public
    procedure AssignImageData( bits: pointer; nWidth, nHeight: integer ); virtual;
    procedure LoadFromFile( const s: TFileName ); virtual;
    constructor Create( AOwner: TComponent ); override;
    destructor Destroy; override;
  published
    property ColorPalette: TColorPalette read FColorPalette write setColorPalette;
    property DataLoaded: boolean read GetDataLoaded write SetDataLoaded;
    property DIBCanvas: TDIBCanvas read FDIBCanvas;
    property DIBLoadFile: TFileName read FFileName write SetFileName;
    property FramesX: integer read FFramesX write FFramesX;
    property FramesY: integer read FFramesY write FFramesY;
    property DIBLoadImage: TImage read FImage write SetImage;
    property Transparent: boolean read FTrans write FTrans;
  end;

{ Property Editors }
  TColorIndexSelectorSS = class( TIntegerProperty )
  private
  protected
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TColorIndexSelectorDC = class( TIntegerProperty )
  private
  protected
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

  TDIBFileNameProperty = class( TPropertyEditor )
  private
  protected
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: string; override;
    procedure SetValue( const Value: string ); override;
  end;

procedure MoveTowardDestination( sprite: TSprite );
procedure MoveThrust( sprite: TSprite );
procedure Register;

implementation

(*********************************************
TSpriteSurface
*********************************************)
constructor TSpriteSurface.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FStretch := false;
  FDIBCanvas := TDIBCanvas.Create;
  FDIBCreated := false;
  Width := 120;
  Height := 120;
  FResized := false;
  lstEngines := TList.Create;
  SortEngines := true;
  {
  if not (csDesigning in ComponentState) then
    ShowMessage( 'Thank you for evaluating TurboSprite 3.0!  Please visit http://www.silicmdr.com for Registration Information.' );
  }
end;

destructor TSpriteSurface.Destroy;
begin
  if FDIBCreated then
    DestroyDIB;
  FDIBCanvas.Free;
  lstEngines.Free;
  inherited Destroy;
end;

procedure TSpriteSurface.Loaded;
begin
  inherited Loaded;
  FDIBCanvas.ColorPalette := FColorPalette;
end;

procedure TSpriteSurface.Notification( AComponent: TComponent; Operation: TOperation );
begin
  inherited Notification( AComponent, Operation );
  if (AComponent = FColorPalette) and (Operation = opRemove) then
    ColorPalette := nil;
  if (AComponent = FBackground) and (Operation = opRemove) then
    FBackground := nil;
end;

procedure TSpriteSurface.setColorPalette( pal: TColorPalette );
begin
  FColorPalette := pal;
  FDIBCanvas.ColorPalette := pal;
end;

procedure TSpriteSurface.Paint;
begin
  if ( csDesigning in ComponentState ) then
  begin
    Canvas.Brush.Style := bsSolid;
    Canvas.Brush.Color := clWhite;
    Canvas.Pen.Color := clBlack;
    Canvas.Rectangle( 0, 0, Width - 1, Height - 1 );
  end
  else
  begin
    FForce := true;
    DIBRefresh;
  end;
end;

procedure TSpriteSurface.CreateDIB;
var
  i: integer;
begin
  if not Assigned( FColorPalette ) then
    raise ETurboSprite.Create( 'ColorPalette Property must be Assigned' );
  FDIBDC := CreateCompatibleDC( 0 );
  with FBitmapInfo.bmheader do
  begin
    biSize := SizeOf( FBitmapInfo.bmheader );
    biWidth := FDIBCanvas.Width;
    biHeight := FDIBCanvas.Height;
    biPlanes := 1;
    biBitCount := 8;
    biCompression := BI_RGB;
    if FDIBCanvas.Orientation = orTopDown then
      biHeight := -biHeight;
    biSizeImage := FDIBCanvas.Size;
    biXPelsPerMeter := GetDeviceCaps( Canvas.Handle, HORZRES ) div GetDeviceCaps( Canvas.Handle, HORZSIZE );
    biYPelsPerMeter := GetDeviceCaps( Canvas.Handle, VERTRES ) div GetDeviceCaps( Canvas.Handle, VERTSIZE );
    biClrUsed := 256;
    biClrImportant := 256;
  end;
  for i := 0 to 255 do
  begin
    FBitmapInfo.bmiColors[i].rgbRed := FColorPalette.PaletteEntry[i].peRed;
    FBitmapInfo.bmiColors[i].rgbGreen := FColorPalette.PaletteEntry[i].peGreen;
    FBitmapInfo.bmiColors[i].rgbBlue := FColorPalette.PaletteEntry[i].peBlue;
    FBitmapInfo.bmiColors[i].rgbReserved := 0;
  end;
  FDIBhandle := CreateDIBSection( FDIBDC, PBitmapInfo(@FBitmapInfo)^, DIB_RGB_COLORS, FDIBbits, nil, 0 );
  FDIBCanvas.Bits := FDIBBits;
  FOldBitmap := SelectObject( FDIBDC, FDIBHandle );
  SelectPalette( FDIBDC, FColorPalette.Palette, false );
  RealizePalette( FDIBDC );
  SelectPalette( Canvas.Handle, FColorPalette.Palette, false );
  RealizePalette( Canvas.Handle );
  FGDICanvas := TCanvas.Create;
  FGDICanvas.Handle := FDIBDC;
  FDIBCreated := true;
end;

procedure TSpriteSurface.DestroyDIB;
begin
  if FOldBitmap <> 0 then
    SelectObject( FDIBDC, FOldBitmap );
  if FDIBHandle <> 0 then
    DeleteObject( FDIBHandle );
  DeleteObject( FDIBDC );
  FDIBCreated := false;
end;

procedure TSpriteSurface.SetBounds( ALeft, ATop, AWidth, AHeight: Integer );
begin
  inherited SetBounds( ALeft, ATop, (((AWidth+3) div 4) * 4), AHeight );
  if FDIBCanvas = nil then
    Exit;
  if ( (csDesigning in ComponentState) ) then
  begin
    FDIBCanvas.Width := Width;
    FDIBCanvas.Height := Height;
  end
  else if not FStretch then
    FResized := true;
  if (not FLockLogical) then
  begin
    FLogicalWidth := Width;
    FLogicalHeight := Height;
  end;
  if Assigned( FOnResize ) then
    FOnResize( self );
end;

function sortProc( Item1, Item2: pointer ): integer;
var
  eng1, eng2: TSpriteEngine;
begin
  eng1 := TSpriteEngine( Item1 );
  eng2 := TSpriteEngine( Item2 );
  if eng1.Priority = eng2.Priority then
    Result := 0
  else if eng1.Priority > eng2.Priority then
    Result := -1
  else
    Result := 1;
end;

procedure TSpriteSurface.DIBRefresh;
var
  i: integer;
  pdc: HDC;
begin
try
  if not FDIBCreated then
    CreateDIB;
  if SortEngines then
  begin
    lstEngines.Sort( sortProc );
    SortEngines := false;
  end;
  if FResized then
  begin
    FResized := false;
    DestroyDIB;
    FDIBCanvas.Width := Width;
    FDIBCanvas.Height := Height;
    CreateDIB;
  end;
  if FAutoBlank then
    if FForce or not FDirty then
      FillBlock( FDIBBits, FDIBCanvas.Width, FDIBCanvas.Height, FDIBCanvas.Width, FDIBCanvas.Height, 0, 0, FAutoColor, FDIBCanvas.Orientation );
  if Assigned( FBackground ) then
    if FForce or not FDirty then
      DumpDIB( FBackground.DIBCanvas.Bits, FDIBBits, FBackground.DIBCanvas.Width,
        FBackground.DIBCanvas.Height, Ord( FDIBCanvas.Orientation ) );
  if Assigned( FBeforeSpriteRender ) then
    FBeforeSpriteRender( self );
  for i := 0 to lstEngines.Count - 1 do
    TSpriteEngine( lstEngines[i] ).ProcessSprites;
  for i := 0 to lstEngines.Count - 1 do
  begin
    TSpriteEngine( lstEngines[i] ).RenderSprites;
    TSpriteEngine( lstEngines[i] ).DetectCollisions;
  end;
  if Assigned( FAfterSpriteRender ) then
    FAfterSpriteRender( self );
  SelectPalette( Canvas.Handle, FColorPalette.Palette, false );
  if ((Width = FDIBCanvas.Width) and (Height = FDIBCanvas.Height)) then
    BitBlt( Canvas.Handle, 0, 0, FDIBCanvas.Width, FDIBCanvas.Height, FDIBDC, 0, 0, SRCCOPY )
  else
    StretchDIBits( Canvas.Handle, 0, 0, Width, Height, 0, 0, FDIBCanvas.Width, FDIBCanvas.Height, FDIBBits, PBitmapInfo(@FBitmapInfo)^, DIB_RGB_COLORS, SRCCOPY );
  FForce := false;
except
end;
end;

procedure TSpriteSurface.setAutoBlank( b: boolean );
begin
  FAutoBlank := b;
  FForce := true;
end;

procedure TSpriteSurface.setAutoBlankColor( n: byte );
begin
  FAutoColor := n;
  FForce := true;
end;

function TSpriteSurface.getMouseX( X: integer ): integer;
var
  xRatio: single;
begin
  if Width = 0 then
  begin
    Result := 0;
    Exit;
  end;
  xRatio := FDIBCanvas.Width / Width;
  Result := Round( X * xRatio + OffsetX * xRatio );
end;

function TSpriteSurface.getMouseY( Y: integer ): integer;
var
  xRatio: single;
begin
 if Height = 0 then
  begin
    Result := 0;
    Exit;
  end;
  xRatio := FDIBCanvas.Height / Height;
  Result := Round( Y * xRatio + OffsetY * xRatio );
end;

procedure TSpriteSurface.MouseDown( Button: TMouseButton; Shift: TShiftState; X, Y: integer );
begin
  inherited MouseDown( Button, Shift, X, Y );
  if Assigned( FMouseDown ) then
    FMouseDown( self, Button, Shift, getMouseX( X ), getMouseY( Y ) );
end;

procedure TSpriteSurface.MouseMove( Shift: TShiftState; X, Y: Integer );
begin
  inherited MouseMove( Shift, X, Y );
  if Assigned( FAbsMouseMove ) then
    FAbsMouseMove( self, Shift, X, Y );
  if Assigned( FMouseMove ) then
    FMouseMove( self, Shift, getMouseX( X ), getMouseY( Y ) );
end;

procedure TSpriteSurface.MouseUp( Button: TMouseButton; Shift: TShiftState; X, Y: integer );
var
  i, n, s: integer;
  en: TSpriteEngine;
  spr: TSprite;
  pt: TPoint;
  rect: TRect;
begin
  pt := Point( getMouseX( X ), getMouseY( Y ) );
  inherited MouseUp( Button, Shift, X, Y );
  if Assigned( FMouseUp ) then
    FMouseUp( self, Button, Shift, pt.X, pt.Y );
  if Assigned( FSpriteClicked ) then
  begin
    for i := 0 to lstEngines.Count - 1 do
    begin
      en := TSpriteEngine( lstEngines[i] );
      if not en.ClickSprites then
        continue;
      n := en.SpriteCount;
      for s := 0 to n - 1 do
      begin
        spr := en.Sprite[s];
        rect := spr.BoundingRect;
        OffsetRect( rect, OffsetX, OffsetY );
        if PtInRect( rect, pt ) then
          if spr.Visible then
          begin
            FSpriteClicked( self, Button, Shift, spr, pt.X, pt.Y );
            Exit;
          end;
      end;
    end;
    FSpriteClicked( self, Button, Shift, nil, pt.X, pt.Y );
  end;
end;

function TSpriteSurface.randomPoint: TPoint;
begin
  Result := Point( Random(LogicalWidth), Random(LogicalHeight) );
end;

procedure TSpriteSurface.synchColorEntries( StartIndex, NbrOfEntries: integer );
begin
  SetDIBColorTable( FDIBDC, StartIndex, NbrOfEntries, (@FColorPalette.pal.logPalette.palpalEntry[StartIndex])^ );
end;

(*********************************************
TRectHolder
*********************************************)
constructor TRectHolder.Create( rectangle: TRect );
begin
  rect := rectangle;
end;

(*********************************************
TSpriteEngine
*********************************************)
constructor TSpriteEngine.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  lstSprites := TList.Create;
  lstNewSprites := TList.Create;
  lstDeadSprites := TList.Create;
  lstDirty := TList.Create;
  lstEngines := TList.Create;
  FClick := true;
end;

destructor TSpriteEngine.Destroy;
begin
  if FSpriteSurface <> nil then
    with FSpriteSurface do
      lstEngines.Delete( lstEngines.IndexOf( self ) );
  lstSprites.Free;
  lstNewSprites.Free;
  lstDeadSprites.Free;
  lstDirty.Free;
  lstEngines.Free;
  inherited Destroy;
end;

procedure TSpriteEngine.ProcessSprites;
var
  i, k, nIndex: integer;
  rTemp: TRect;
begin
  if not Assigned( FSpriteSurface ) then
    Exit;
  bDrawAll := FSpriteSurface.FForce or not FSpriteSurface.FDirty;

{ 1 - Add new sprites into the sprite master list }
  while lstNewSprites.Count > 0 do
  begin
    nIndex := -1;
    for i := 0 to lstSprites.Count - 1 do
      if TSprite( lstNewSprites[0] ).Priority <= TSprite( lstSprites[i] ).Priority then
      begin
        nIndex := i;
        Break;
      end;
    if nIndex = -1 then
      lstSprites.Add( lstNewSprites[0] )
    else
      lstSprites.Insert( nIndex, lstNewSprites[0] );
    lstNewSprites.Delete( 0 );
  end;

{ 2 - Move all sprites }
  for i := 0 to lstSprites.Count - 1 do
  with TSprite( lstSprites[i] ) do
    Move;

{ 3 - Capture dirty rectangles of sprites that have moved }
  if not bDrawAll then
  begin
    for i := 0 to lstSprites.Count - 1 do
      with TSprite( lstSprites[i] ) do
      begin
        if Moved then
          lstDirty.Add( TRectHolder.Create( GetDirtyRect ) );
      end;

{ 4 - Refresh the backgrounds of all sprites that intersect dirty rectangles }
    for i := 0 to lstSprites.Count - 1 do
      with TSprite( lstSprites[i] ) do
      begin
        for k := 0 to lstDirty.Count - 1 do
          if IntersectRect( rTemp, GetDirtyRect, TRectHolder( lstDirty[k] ).rect ) then
          begin
            FDirty := TRUE;
            RefreshBackground;
            Break;
          end;
      end;
  end;

{ 5 - Remove dead sprites from the master list }
  while lstDeadSprites.Count > 0 do
  begin
    nIndex := lstSprites.IndexOf( lstDeadSprites[0] );
    if nIndex >= 0 then
    begin
      TSprite( lstSprites[nIndex] ).Free;
      lstSprites.Delete( nIndex );
    end
    else
    begin
      nIndex := lstNewSprites.IndexOf( lstDeadSprites[0] );
      if nIndex >= 0 then
      begin
        TSprite( lstNewSprites[nIndex] ).Free;
        lstNewSprites.Delete( nIndex );
      end;
    end;
    lstDeadSprites.Delete( 0 );
  end;
  
end;

procedure TSpriteEngine.RenderSprites;
var
  i: integer;
  rect: TRect;
begin
  if not Assigned( FSpriteSurface ) then
    Exit;
  if FSpriteSurface.DIBCanvas.Bits = nil then
    Exit;
  if bDrawAll then
  begin
    for i := lstSprites.Count - 1 downto 0 do
    begin
      with TSprite( lstSprites[i] ) do
      begin
        if IntersectRect( rect, BoundingRect, SpriteSurface.ClientRect ) then
          if Visible then
          begin
            Render;
            Application.ProcessMessages;
          end;
      end;
    end;
  end
  else
  begin
    for i := lstSprites.Count - 1 downto 0 do
      with TSprite( lstSprites[i] ) do
      begin
        if FDirty then
          Render;
      end;
    while lstDirty.Count > 0 do
    begin
      TRectHolder( lstDirty[0] ).Free;
      lstDirty.Delete( 0 );
    end;
  end;
end;

function TSpriteEngine.GetSprite( n: integer ): TSprite;
begin
  Result := TSprite( lstSprites[n] );
end;

function TSpriteEngine.GetSpriteCount: integer;
begin
  Result := lstSprites.Count;
end;

procedure TSpriteEngine.SetExpectedSprites( n: integer );
begin
  FExpectedSprites := n;
  if n > lstSprites.Capacity then
     lstSprites.Capacity := n;
end;

procedure TSpriteEngine.AddSprite( spr: TSprite );
begin
  spr.FSpriteSurface := FSpriteSurface;
  spr.FSpriteEngine := self;
  lstNewSprites.Add( spr );
  if FSpriteSurface <> nil then
    FSpriteSurface.FForce := true;
end;

procedure TSpriteEngine.RemoveSprite( spr: TSprite );
begin
  spr.Dead := TRUE;
  lstDeadSprites.Add( spr );
  FSpriteSurface.FForce := true;
end;

procedure TSpriteEngine.removeAllSprites;
var
  n: integer;
begin
  for n := lstSprites.Count - 1 downto 0 do
    removeSprite( Sprite[n] );
end;

procedure TSpriteEngine.ChangeSpritePriority( spr: TSprite; n: integer );
begin
  lstDeadSprites.Add( spr );
  spr.Priority := n;
  lstNewSprites.Add( spr );
end;

procedure TSpriteEngine.CollisionDetection( se: TSpriteEngine );
var
  i, k: integer;
  spr, sprTarget: TSprite;
  rTemp: TRect;
begin
  if not Assigned( FCollision ) then
    Exit;
  for i := 0 to lstSprites.Count - 1 do
  begin
    spr := TSprite( lstSprites[i] );
    if se = self then
    begin
      for k := i + 1 to lstSprites.Count - 1 do
      begin
        sprTarget := TSprite( lstSprites[k] );
        if IntersectRect( rTemp, spr.CollisionRect, sprTarget.CollisionRect ) then
           FCollision( self, spr, sprTarget );
      end;
    end
    else
    begin
      for k := 0 to se.lstSprites.Count - 1 do
      begin
        sprTarget := TSprite( se.lstSprites[k] );
        if IntersectRect( rTemp, spr.CollisionRect, sprTarget.CollisionRect ) then
           FCollision( self, spr, sprTarget );
      end;
    end;
  end;
end;

procedure TSpriteEngine.Notification( AComponent: TComponent; Operation: TOperation );
begin
  inherited Notification( AComponent, Operation );
  if AComponent = FSpriteSurface then
     if Operation = opRemove then
        FSpriteSurface := nil;
end;

procedure TSpriteEngine.setSpriteSurface( ss: TSpriteSurface );
begin
  FSpriteSurface := ss;
  if ( ss <> nil ) then
    ss.lstEngines.Add( self );
end;

procedure TSpriteEngine.AddCollisionEngine( engine: TSpriteEngine );
begin
  lstEngines.Add( engine );
end;

procedure TSpriteEngine.RemoveCollisionEngine( engine: TSpriteEngine );
var
  i: integer;
begin
  i := lstEngines.IndexOf( engine );
  if ( i >= 0 ) then
    lstEngines.Delete( i );
end;

procedure TSpriteEngine.DetectCollisions;
var
  i: integer;
begin
  for i := 0 to lstEngines.Count - 1 do
    CollisionDetection( TSpriteEngine( lstEngines[i] ) );
end;

procedure TSpriteEngine.setPri( n: integer );
begin
  FPri := n;
  if Assigned( FSpriteSurface ) then
    FSpriteSurface.SortEngines := true;
end;

(*********************************************
TSprite
*********************************************)
constructor TSprite.Create( MoveProc: TMoveProc );
begin
  FSpriteSurface := nil;
  Priority := 1;
  X := 0.0;
  Y := 0.0;
  FSpeed := 0.0;
  ptDestination := Point( 0, 0 );
  FLastDest.X := 0;
  FLastDest.Y := 0;
  FVisible := TRUE;
  FMoveProc := MoveProc;
end;

procedure TSprite.Move;
var
  FLastX, FLastY: single;
begin
  FMoved := false;
  FLastX := X;
  FLastY := Y;
  case FSpinType of
    spinClockWise:
      Angle := Angle + FSpinSpeed;
    spinCounterClockWise:
      Angle := Angle - FSpinSpeed;
  end;
  if Angle <> OldAngle then
    FMoved := true;
  OldAngle := Angle;
  if Assigned( FMoveProc ) then
    FMoveProc( self );
  with FSpriteSurface do
  begin
    if FWrapHorz then
    begin
      while ( X < 0 ) do
        X := X + FLogicalWidth;
      while ( X >= FLogicalWidth ) do
        X := X - FLogicalWidth;
    end;
    if FWrapVert then
    begin
      while ( Y < 0 ) do
        Y := Y + FLogicalHeight;
      while ( Y >= FLogicalHeight ) do
        Y := Y - FLogicalHeight;
    end;
    ptPhysical.X := Round( X );
    ptPhysical.Y := Round( Y );
    Dec( ptPhysical.X, FOffsetX );
    Dec( ptPhysical.Y, FOffsetY );
  end;
  if not FMoved then
    FMoved := (X <> FLastX) or (Y <> FLastY);
end;

function TSprite.getCorrectedAngle: integer;
begin
  Result := IntToDegree(FAngle - 90 - 180);
end;

procedure TSprite.SetDestination( const pt: TPoint );
begin
  ptDestination.X := pt.X;
  ptDestination.Y := pt.Y;
end;

procedure TSprite.SetSpeed( const x: single );
begin
  FSpeed := x;
  if ( FSpeed < 0.0 ) then
    FSpeed := 0.0;
end;

function TSprite.FudgedDistance( s: TSprite ): single;
begin
  Result := Abs( X - s.X ) + Abs( Y - s.Y );
end;

procedure TSprite.Render;
begin
  ptLastDrawn := ptPhysical;
end;

function TSprite.GetDirtyRect: TRect;
var
  rectOld, rectNew, rectUnion: TRect;
begin
  rectOld := Rect( ptLastDrawn.X - FWidth2, ptLastDrawn.Y - FHeight2, ptLastDrawn.X + FWidth2, ptLastDrawn.Y + FHeight2 );
  rectNew := Rect( ptPhysical.X - FWidth2, ptPhysical.Y - FHeight2, ptPhysical.X + FWidth2, ptPhysical.Y + FHeight2 );
  UnionRect( rectUnion, rectOld, rectNew );
  Result := rectUnion;
end;

procedure TSprite.RefreshBackground;
var
  ptDest: TPoint;
  rectSource: TRect;
begin
  if Visible then
  begin
    if Assigned( FSpriteSurface.BackgroundDIB ) then
    begin
      with rectSource do
      begin
        Left := ptLastDrawn.X - FWidth2;
        Top := ptLastDrawn.Y - FHeight2;
        Right := Left + Width;
        Bottom := Top + Height;
      end;
      FSpriteSurface.DIBCanvas.CopyRect( rectSource, FSpriteSurface.BackgroundDIB.DIBCanvas, rectSource, false );
    end
    else
    begin
      FSpriteSurface.DIBCanvas.BrushColorIndex := FSpriteSurface.AutoBlankColorIndex;
      FSpriteSurface.DIBCanvas.FillRect( Rect( ptLastDrawn.X - FWidth2, ptLastDrawn.Y - FHeight2, ptLastDrawn.X + FWidth2 - 1, ptLastDrawn.Y + FHeight2 - 1 ) );
    end;
  end;
end;

procedure TSprite.SetWidth( n: integer );
begin
  FWidth := n;
  FWidth2 := n div 2;
end;

procedure TSprite.SetHeight( n: integer );
begin
  FHeight := n;
  FHeight2 := n div 2;
end;

function TSprite.GetBoundingRect: TRect;
begin
  Result := Rect( ptPhysical.X - FWidth2, ptPhysical.Y - FHeight2, ptPhysical.X + FWidth2 + 1, ptPhysical.Y + FHeight2 + 1 );
end;

function TSprite.GetCollisionRect: TRect;
begin
  Result := Rect( ptPhysical.X - FWidth2 + FMarginLeft, ptPhysical.Y - FHeight2 + FMarginTop, ptPhysical.X + FWidth2 + 1 - FMarginRight, ptPhysical.Y + FHeight2 + 1 - FMarginBottom );
end;

procedure TSprite.SetDead( b: boolean );
begin
  if b <> bKill then
  begin
    bKill := b;
    if b then
      if FSpriteEngine <> nil then
        FSpriteEngine.RemoveSprite( self );
  end;
end;

procedure TSprite.setAngle( n: integer );
begin
  FAngle := IntToDegree(n);
  FMoved := true;
end;

function TSprite.destinationChanged: boolean;
begin
  Result := (ptDestination.X <> FLastDest.X) or (ptDestination.Y <> FLastDest.Y);
  if Result then
  begin
    FLastDest.X := ptDestination.X;
    FLastDest.Y := ptDestination.Y;
  end;
end;

function TSprite.speedChanged: boolean;
begin
  Result := FLastSpeed <> FSpeed;
  if Result then
    FLastSpeed := FSpeed;
end;

procedure TSprite.Stop;
begin
  ptDestination.X := Round( X );
  ptDestination.Y := Round( Y );
  FSpeed := 0;
  FSpeedX := 0;
  FSpeedY := 0;
  FThrust := 0;
end;

function TSprite.getPosition: TPoint;
begin
  Result := Point( Trunc(X), Trunc(Y) );
end;

procedure TSprite.setPosition( pt: TPoint );
begin
  X := pt.X;
  Y := pt.Y;
end;

(*********************************************
TDIB
*********************************************)
constructor TDIB.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  FDIBCanvas := TDIBCanvas.Create;
  FDIBCanvas.ColorPalette := FColorPalette;
  pDIBBits := nil;
end;

destructor TDIB.Destroy;
begin
  CleanUp;
  FDIBCanvas.Free;
  inherited Destroy;
end;

procedure TDIB.CleanUp;
begin
  if pDIBBits <> nil then
  begin
    VirtualFree( pDIBBits, FDIBCanvas.Size, MEM_RELEASE or MEM_DECOMMIT );
    pDIBBits := nil;
    FFileName := '';
  end;
end;

procedure TDIB.LoadFromFile( const s: TFileName );
var
  fileDIB: TFileStream;
  pbmFileHeader: PBitmapFileHeader;
  pbmInfoHeader: PBitmapInfoHeader;
  pBits: PByte;
  pFile, p: pointer;
  nSize, ImageSize: integer;
begin
  CleanUp;
try
  fileDIB := TFileStream.Create( s, fmOpenRead );
  nSize := fileDIB.Size;
  pFile := VirtualAlloc( nil, nSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );
  fileDIB.Read( pFile^, nSize );
  fileDIB.Free;

{ Make sure this is a valid BMP file }
  pbmFileHeader := pFile;
  if pbmFileHeader^.bfType <> MakeWord( Ord( 'B' ), Ord( 'M' ) ) then
     raise ETurboSprite.Create( 'Not a Bitmap file' );

{ Capture the offset to the BitmapInfoHeader }
  pBits := pFile;
  Inc( pBits, SizeOf( TBitmapFileHeader ) );
  p := pBits;
  pbmInfoHeader := p;

{ Capture the offset to the DIB data itself }
  pBits := pFile;
  Inc( pBits, pbmFileHeader^.bfOffBits );

{ Make sure the DIB is of the correct format }
  if pbmInfoHeader^.biPlanes <> 1 then
     raise ETurboSprite.Create( 'biPlanes <> 1' );
  if pbmInfoHeader^.biBitCount <> 8 then
     raise ETurboSprite.Create( 'Only 256 color DIBs supported' );
  if pbmInfoHeader^.biCompression <> 0 then
     raise ETurboSprite.Create( 'RLE DIBs not yet supported' );

{ Allocate memory for DIB data }
  ImageSize := pbmInfoHeader^.biWidth * pbmInfoHeader^.biHeight;
  pDIBBits := VirtualAlloc( nil, ImageSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );

{ Copy from disk image into buffer }
  CopyMemory( pDIBBits, pBits, ImageSize );

{ Move values into the DIBCanvas }
  with FDIBCanvas do
  begin
    Width := pbmInfoHeader^.biWidth;
    Height := pbmInfoHeader^.biHeight;
    Bits := pDIBBits;
    Orientation := orBottomUp;
  end;

finally
  VirtualFree( pFile, nSize, MEM_RELEASE or MEM_DECOMMIT );
end;
end;

procedure TDIB.SetFileName( const sName: TFileName );
begin
  if sName <> '' then
  begin
    LoadFromFile( sName );
    if csDesigning in ComponentState then
      ShowMessage( 'DIB Data Loaded from file ' + sName );
  end;
end;

procedure TDIB.LoadFromBitmap( bm: Graphics.TBitmap );
begin
  bm.SaveToFile( '~~TEMP.BMP' );
  LoadFromFile( '~~TEMP.BMP' );
  FFileName := '';
  DeleteFile( '~~TEMP.BMP' );
end;

procedure TDIB.SetImage( im: TImage );
begin
  if im <> nil then
  begin
    LoadFromBitmap( im.Picture.Bitmap );
    if csDesigning in ComponentState then
      ShowMessage( 'DIB Data Loaded from TImage ' + im.Name );
  end;
end;

procedure TDIB.ReadImage( Stream: TStream );
var
  nWidth, nSize, nHeight: integer;
begin
  CleanUp;
  Stream.Read( nWidth, SizeOf( integer ) );
  Stream.Read( nHeight, SizeOf( integer ) );
  nSize := nWidth * nHeight;
  if nSize > 0 then
  begin
    pDIBBits := VirtualAlloc( nil, nSize, MEM_RESERVE or MEM_COMMIT, PAGE_READWRITE );
    Stream.Read( pDIBBits^, nSize );
    with FDIBCanvas do
    begin
      Width := nWidth;
      Height := nHeight;
      Bits := pDIBBits;
      Orientation := orBottomUp;
    end;
  end;
end;

procedure TDIB.WriteImage( Stream: TStream );
begin
  Stream.Write( FDIBCanvas.Width, SizeOf( integer ) );
  Stream.Write( FDIBCanvas.Height, SizeOf( integer ) );
  if (FDIBCanvas.Size) > 0 then
     Stream.Write( FDIBCanvas.Bits^, FDIBCanvas.Size );
end;

procedure TDIB.DefineProperties( Filer: TFiler );
begin
  inherited DefineProperties( Filer );
  Filer.DefineBinaryProperty( 'ImageData', ReadImage, WriteImage, true );
end;

function TDIB.GetDataLoaded: boolean;
begin
  Result := FDIBCanvas.Size > 0;
end;

procedure TDIB.SetDataLoaded( b: boolean );
begin
  if not b then
    CleanUp;
end;

procedure TDIB.setColorPalette( pal: TColorPalette );
begin
  FColorPalette := pal;
  if FDIBCanvas <> nil then
    FDIBCanvas.ColorPalette := pal;
end;

procedure TDIB.Notification( AComponent: TComponent; Operation: TOperation );
begin
  inherited Notification( AComponent, Operation );
  if (AComponent = FColorPalette) and (Operation = opRemove) then
  begin
    FColorPalette := nil;
    FDIBCanvas.ColorPalette := nil;
  end;
end;

procedure TDIB.Loaded;
begin
  inherited Loaded;
  FDIBCanvas.ColorPalette := FColorPalette;
end;

procedure TDIB.AssignImageData( bits: pointer; nWidth, nHeight: integer );
begin
  with DIBCanvas do
  begin
    DeAllocate;
    Bits := bits;
    Width := nWidth;
    Height := nHeight;
  end;
end;

(*********************************************
TColorIndexSelector Property Editors
*********************************************)
procedure TColorIndexSelectorSS.Edit;
var
  cpSelf: TColorPalette;
  oldFlag: TPalEntryFlag;
begin
  cpSelf := TSpriteSurface( GetComponent( 0 ) ).ColorPalette;
  if cpSelf = nil then
  begin
    ShowMessage( 'First Assign a TColorPalette' );
    Exit;
  end;
  with TfrmColorIndexSelector.Create( nil ) do
  begin
    oldFlag := cpSelf.PalEntryFlag;
    cpSelf.PalEntryFlag := pcReserved;
    ColorPalette1.Assign( cpSelf );
    cpSelf.PalEntryFlag := oldFlag;
    nIndex := GetOrdValue;
    if ShowModal = mrOk then
      SetOrdValue( nIndex );
    Release;
  end;
end;

function TColorIndexSelectorSS.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TColorIndexSelectorDC.Edit;
var
  cpSelf: TColorPalette;
  oldFlag: TPalEntryFlag;
begin
  cpSelf := TDIBCanvas( GetComponent( 0 ) ).ColorPalette;
  if cpSelf = nil then
  begin
    ShowMessage( 'First Assign a TColorPalette' );
    Exit;
  end;
  with TfrmColorIndexSelector.Create( nil ) do
  begin
    oldFlag := cpSelf.PalEntryFlag;
    cpSelf.PalEntryFlag := pcReserved;
    ColorPalette1.Assign( cpSelf );
    cpSelf.PalEntryFlag := oldFlag;
    nIndex := GetOrdValue;
    if ShowModal = mrOk then
      SetOrdValue( nIndex );
    Release;
  end;
end;

function TColorIndexSelectorDC.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TDIBFileNameProperty.Edit;
begin
  with TfrmDIBFileLoader.Create( nil ) do
  begin
    if ShowModal = mrOk then
    begin
      if FileListBox1.ItemIndex >= 0 then
        SetStrValue( FileListBox1.FileName );
    end;
    Release;
  end;
end;

function TDIBFileNameProperty.GetValue: string;
begin
  Result := GetStrValue;
end;

procedure TDIBFileNameProperty.SetValue( const Value: string );
begin
  SetStrValue( Value );
end;

function TDIBFileNameProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

(*********************************************
MoveProcs
*********************************************)
procedure MoveTowardDestination( sprite: TSprite );
var
  Dist: single;
  xPct, yPct: single;
  bMoveX, bMoveY: boolean;
begin
  with sprite do
  begin
    if destinationChanged or speedChanged then
    begin
      Dist := Abs( ptDestination.X - X ) + Abs( ptDestination.Y - Y );
      if ( Dist > 0.0 ) then
      begin
        xPct := Abs( ptDestination.X - X ) / Dist;
        yPct := Abs( ptDestination.Y - Y ) / Dist;
        SpeedX := Speed * xPct;
        SpeedY := Speed * yPct;
        if ( ptDestination.X < X ) then
          SpeedX := -SpeedX;
        if ( ptDestination.Y < Y ) then
          SpeedY := -SpeedY;
      end
      else
      begin
        SpeedX := Speed / 2.0;
        SpeedY := Speed / 2.0;
      end;
    end;
    bMoveX := true;
    bMoveY := true;
    if ( StopAtDestination ) then
    begin
try
      bMoveX := Abs( X - ptDestination.X ) > Speed;
      bMoveY := Abs( Y - ptDestination.Y ) > Speed;
except
end;
    end;
    if bMoveX then
      X := X + SpeedX
    else
      X := ptDestination.X;
    if bMoveY then
      Y := Y + SpeedY
    else
      Y := ptDestination.Y;
  end;
end;

procedure MoveThrust( sprite: TSprite );
var
  VectorX, VectorY: single;
begin
  with sprite do
  begin
    if Thrust > 0 then
    begin
      VectorX := Thrust * lkupCos[correctedAngle];
      VectorY := Thrust * lkupSin[correctedAngle];
      SpeedX := SpeedX + VectorX;
      SpeedY := SpeedY + VectorY;
    end;
    if (Friction > 0) and (Thrust <= 0) then
    begin
      if SpeedX > 0 then
        SpeedX := SpeedX - Friction;
      if SpeedX < 0 then
        SpeedX := SpeedX + Friction;
      if SpeedY > 0 then
        SpeedY := SpeedY - Friction;
      if SpeedY < 0 then
        SpeedY := SpeedY + Friction;
    end;
    if SpeedX > MaximumSpeed then
      SpeedX := MaximumSpeed;
    if SpeedY > MaximumSpeed then
      SpeedY := MaximumSpeed;
    if SpeedX < -MaximumSpeed then
      SpeedX := -MaximumSpeed;
    if SpeedY < -MaximumSpeed then
      SpeedY := -MaximumSpeed;
    if SpeedX <> 0 then
      X := X + SpeedX;
    if SpeedY <> 0 then
      Y := Y + SpeedY;
  end;
end;

procedure Register;
begin
  RegisterPropertyEditor( TypeInfo( byte ), TSpriteSurface, 'AutoBlankColorIndex', TColorIndexSelectorSS );
  RegisterPropertyEditor( TypeInfo( byte ), TDIBCanvas, 'PenColorIndex', TColorIndexSelectorDC );
  RegisterPropertyEditor( TypeInfo( byte ), TDIBCanvas, 'BrushColorIndex', TColorIndexSelectorDC );
  RegisterPropertyEditor( TypeInfo( byte ), TDIBCanvas, 'TransparentColorIndex', TColorIndexSelectorDC );
  RegisterPropertyEditor( TypeInfo( TFileName ), TDIB, 'DIBLoadFile', TDIBFileNameProperty );
  RegisterComponents( 'TurboSprite', [TSpriteSurface,TSpriteEngine,TDIB] );
  ColorPalette.Register;
end;

end.
